using System;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Permissions;
using Microsoft.SharePoint;
using System.Globalization;
using Microsoft.SharePoint.Administration;

namespace Code6587EN.Ch04.Features.SiteCollectionBranding
{
    [Guid("c9c76e9e-550d-4b4e-b88b-4399b30d97b5")]
    public class SiteCollectionBrandingEventReceiver : SPFeatureReceiver
    {
        private static Guid BrandingFeatureId = new Guid("1150dec7-4af6-44d8-b241-d976d26b723c");
        private static string FormatJobName = "Custom Branding Job_{0}";

        public override void FeatureActivated(SPFeatureReceiverProperties properties)
        {
            // Get the Site Collection the feature was activated on
            using (var site = properties.Feature.Parent as SPSite)
            {
                // Ensure the Site Collection is not null
                if (site != null)
                {
                    // Iterate through each Site in the Site Collection
                    foreach (SPWeb web in site.AllWebs)
                    {
                        // Check to ensure the Site is not null and it exists
                        if (web != null && web.Exists)
                        {
                            // If the feature is not activated on the Site, activate it
                            if (web.Features[BrandingFeatureId] == null)
                                web.Features.Add(BrandingFeatureId);
                        }
                    }

                    // When passing variables into the RunWithElevatedPrivileges delegate
                    // it is import to only pass primitive types (Guids, strings, etc.).
                    // Passing SharePoint objects can yield unexpected results.
                    var jobName = string.Format(CultureInfo.InvariantCulture, FormatJobName, site.ID.ToString());
                    var webAppId = site.WebApplication.Id;
                    var siteId = site.ID;

                    // Create a delegate to run with elevated privileges.  This allows the
                    // code to be run as the Service Account instead of the current user.
                    SPSecurity.RunWithElevatedPrivileges(delegate()
                    {
                        // Get the Web Application
                        var webApplication = SPWebService.ContentService.WebApplications[webAppId];

                        // Remove any existing timer jobs with the same name for this Site Collection
                        foreach (SPJobDefinition job in webApplication.JobDefinitions.Where(p => p.Name == jobName))
                            job.Delete();                        
                        
                        // Create a new instance of the Branding Timer Job
                        var brandingJob = new BrandingTimerJob(webApplication, jobName);
                        brandingJob.Properties.Add("SiteId", siteId);
                        brandingJob.Schedule = new SPDailySchedule() { BeginHour = 1 };
                        brandingJob.Update();
                    });
                }
            }
        }

        public override void FeatureDeactivating(SPFeatureReceiverProperties properties)
        {
            // Get the Site Collection the feature was activated on
            using (var site = properties.Feature.Parent as SPSite)
            {
                // Ensure the Site Collection is not null
                if (site != null)
                {
                    // When passing variables into the RunWithElevatedPrivileges delegate
                    // it is import to only pass primitive types (Guids, strings, etc.).
                    // Passing SharePoint objects can yield unexpected results.
                    var jobName = string.Format(CultureInfo.InvariantCulture, FormatJobName, site.ID.ToString());
                    var webAppId = site.WebApplication.Id;

                    // Create a delegate to run with elevated privileges.  This allows the
                    // code to be run as the Service Account instead of the current user.
                    SPSecurity.RunWithElevatedPrivileges(delegate()
                    {
                        // Get the Web Application and remove any jobs that match the job name
                        var webApplication = SPWebService.ContentService.WebApplications[webAppId];
                        foreach (SPJobDefinition job in webApplication.JobDefinitions.Where(p => p.Name == jobName))                        
                            job.Delete();                        
                    });
                }
            }
        }
    }
}
